/*
Half-Life MAP viewing utility.
Copyright (C) 2003  Ryan Samuel Gregg

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "stdafx.h"
#include "TextureManager.h"

CTextureManager::CTextureManager(CConfig *Config, CRichTextBox *txtConsole)
{
	this->Config = Config;
	this->txtConsole = txtConsole;

	Textures = new ArrayList();
}

int CTextureManager::AddTexture(String *TextureName)
{
	TextureName = TextureName->ToLower();

	for(int i = 0; i < Textures->Count; i++)
	{
		if(static_cast<Texture*>(Textures->get_Item(i))->TextureName->Equals(TextureName))
		{
			return i;
		}
	}

	Textures->Add(new Texture(TextureName));
	return Textures->Count - 1;
}

ArrayList *CTextureManager::GetTextures()
{
	return Textures;
}

Texture *CTextureManager::GetTexture(int iIndex)
{
	return static_cast<Texture*>(Textures->get_Item(iIndex));
}

unsigned int CTextureManager::GetID(int iIndex)
{
	return static_cast<Texture*>(Textures->get_Item(iIndex))->ID;
}

unsigned long CTextureManager::GetWidth(int iIndex)
{
	return static_cast<Texture*>(Textures->get_Item(iIndex))->Width;
}

unsigned long CTextureManager::GetHeight(int iIndex)
{
	return static_cast<Texture*>(Textures->get_Item(iIndex))->Height;
}

String *CTextureManager::GetName(int iIndex)
{
	return static_cast<Texture*>(Textures->get_Item(iIndex))->TextureName;
}

void CTextureManager::LoadTextures(String *WADString)
{
	int iTexturesLoaded = 0, iTotalTexturesLoaded = 0;
	WADString = WADString->Replace("../", "");
	WADString = WADString->Replace('/', '\\');
	String *WAD, *WADs[] = WADString->Split(S";"->ToCharArray());

	bool bWADFound;
	LPCTSTR	szDrive= NULL;
	String *Drives[] = System::IO::Directory::GetLogicalDrives();

	for(int i = 0; i < Drives->Length; i++)
	{
		szDrive = (LPCTSTR)(System::Runtime::InteropServices::Marshal::StringToHGlobalAnsi(Drives[i]).ToPointer());

		if(GetDriveType(szDrive) != DRIVE_FIXED)
			Drives[i] = "";

		System::Runtime::InteropServices::Marshal::FreeHGlobal((IntPtr((void*)szDrive)));
	}

	for(int i = 0; i < WADs->Count; i++)
	{
		bWADFound = false;

		//CWADLoader::LoadWADFile(String::Concat(Config->sHalfLifePath, WADs[i]), Textures);
		WAD = String::Concat(Config->sHalfLifePath, WADs[i])->Replace("\\\\", "\\");

		bWADFound = System::IO::File::Exists(WAD);

		if(!bWADFound)
		{
			for(int j = 0; j < Drives->Length; j++)
			{
				if(Drives[j]->Length == 0)
					continue;

				WAD = String::Concat(Drives[j], WADs[i])->Replace("\\\\", "\\");
				bWADFound = System::IO::File::Exists(WAD);

				if(bWADFound)
					break;
			}
		}

		if(bWADFound)
		{
			txtConsole->Print(String::Concat(S"Loading ", WAD, S"..."), Color::Gray);
			CWADLoader::LoadWADFile(WAD, Textures, &iTexturesLoaded);
			txtConsole->Print(String::Concat(iTexturesLoaded.ToString(), S" textures loaded.", S"\n"), Color::Gray);

			iTotalTexturesLoaded += iTexturesLoaded;
		}
		else
		{
			txtConsole->Print(String::Concat(WADs[i], S" not found.", S"\n"), Color::Red);
		}
	}
	
	txtConsole->Print(String::Concat(iTotalTexturesLoaded.ToString(), S" out of ", Textures->Count.ToString(), S" textures loaded.", S"\n"), Color::Green);

	UpdateTextureFilter();
}

void CTextureManager::UpdateTextureFilter()
{
	Texture *Tex;
	int iMaxAnisotropy = 0;

	if(Config->eTextureFilter == TextureFilter::MipAnisotropic)
	{
		glGetIntegerv(GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT, &iMaxAnisotropy);
	}

	glEnable(GL_TEXTURE_2D);

	for(int i = 0; i < Textures->Count; i++)
	{
		Tex = static_cast<Texture*>(Textures->get_Item(i));
		if(Tex->Loaded)
		{
			glBindTexture(GL_TEXTURE_2D, Tex->ID);
			switch(Config->eTextureFilter)
			{
				case TextureFilter::Nearest:
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, 0);
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
					break;
				case TextureFilter::Linear:
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, 0);
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
					break;
				case TextureFilter::MipNearest:
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, 0);
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_NEAREST);
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST_MIPMAP_NEAREST);
					break;
				case TextureFilter::MipLinear:
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, 0);
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_LINEAR);
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST_MIPMAP_LINEAR);
					break;
				case TextureFilter::MipBilinear:
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, 0);
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_NEAREST);
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR_MIPMAP_NEAREST);
					break;
				case TextureFilter::MipTrilinear:
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, 0);
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR_MIPMAP_LINEAR);
					break;
				case TextureFilter::MipIsotropic:
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, 1);
					break;
				case TextureFilter::MipAnisotropic:
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, iMaxAnisotropy);
					break;
				/*default:
					int iMaxAnisotropy;
					glGetIntegerv(GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT, &iMaxAnisotropy);
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
					glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, iMaxAnisotropy);
					break;*/
			}
		}
	}
}

void CTextureManager::DestroyTextures()
{
	Texture *Tex;
	unsigned int ID;
	int iCount = 0;
	for(int i = 0; i < Textures->Count; i++)
	{
		Tex = static_cast<Texture*>(Textures->get_Item(i));
		if(Tex->Loaded)
		{
			ID = Tex->ID;
			glDeleteTextures(1, &ID);
			Tex->ID = ID;
			iCount++;
		}
	}

	txtConsole->Print(String::Concat(iCount.ToString(), S" textures destroyed.", S"\n"), Color::Green);

	Textures->Clear();
}